/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var advanced_1 = require("report-errors/advanced");
var crypto_1 = require("crypto");
var path = require("path");
var fs = require("fs");
function writeFile(info, outputDir) {
    var input = info.getStructuredStack();
    var notActionableFrames = [];
    var actionableFrames = [];
    var actionable = false;
    input.Stack.forEach(function (frame) {
        if (!actionable) {
            actionable = isFrameActionable(frame);
        }
        if (actionable) {
            actionableFrames.push(frame);
        }
        else {
            notActionableFrames.push(frame);
        }
    });
    if (actionableFrames.length === 0) {
        actionableFrames = notActionableFrames;
        notActionableFrames = [];
    }
    var WLB = advanced_1.WATSON_LINE_BREAK;
    var extraFrames = "";
    for (var i = 0; i < notActionableFrames.length; i++) {
        extraFrames += "ExtraFrame" + i + "=" + frameToString(notActionableFrames[i]) + WLB;
    }
    var stackTrace = "" + firstFrameToString(actionableFrames[0]) + WLB;
    for (var i = 1; i < actionableFrames.length; i++) {
        stackTrace += "           ms-appx://" + frameToString(actionableFrames[i]) + WLB;
    }
    var stackHash = crypto_1.createHmac("md5", "")
        .update(stackTrace)
        .digest("hex");
    var fullStackTrace = ("ErrorTypeText=" + input.ErrorType + WLB) +
        ("ErrorDescription=" + input.ErrorMessage + WLB) +
        ("ErrorNumber=0" + WLB) +
        ("SourceFile=" + actionableFrames[0].Package.name + WLB) +
        ("PackageVersion=" + actionableFrames[0].Package.version + WLB) +
        ("Line=" + actionableFrames[0].line + WLB) +
        ("Character=" + actionableFrames[0].column + WLB) +
        extraFrames +
        ("StackTrace=" + stackTrace) +
        ("StackTraceHash=" + stackHash);
    if (info.errorType) {
        fullStackTrace += WLB + "ErrorContext=" + info.errorType;
    }
    if (info.packageInfo.branch) {
        fullStackTrace += WLB + "CodeBranch=" + info.packageInfo.branch;
    }
    var filePath = path.join(outputDir, "ErrorInfo." + stackHash + ".txt");
    fs.writeFileSync(filePath, fullStackTrace);
    return {
        reportFilePath: filePath,
        sourceFile: actionableFrames[0].RelativePath,
        line: actionableFrames[0].line,
        column: actionableFrames[0].column,
        stackHash: stackHash
    };
}
exports.writeFile = writeFile;
function isFrameActionable(frame) {
    if (frame.Package.name === "node") {
        return false;
    }
    if (frame.extra && (frame.extra.isEval || frame.extra.failedToParse)) {
        return false;
    }
    if (frame.RelativePath === "native" || frame.RelativePath === "unknown" || frame.Package.name === "UNKNOWN") {
        return false;
    }
    if (frame.RelativePath === "UNSUPPORTED") {
        return false;
    }
    return true;
}
function firstFrameToString(frame) {
    return "ms-appx://" + frame.RelativePath + ":0:0           " + frame.FunctionName;
}
function frameToString(frame) {
    var evalFrame = frame.extra && frame.extra.isEval;
    if (frame.Package) {
        return ("" + (evalFrame ? "eval" : frame.Package.name)) +
            ((evalFrame ? "" : "@" + frame.Package.version) + ":" + frame.line + ":" + frame.column + "           ") +
            (frame.RelativePath + "." + frame.FunctionName);
    }
    else {
        return frame.RelativePath + ":" + frame.line + ":" + frame.column + "           " + frame.FunctionName;
    }
}
function setParamsWWAJSE(knownParams, thisPackage, exe) {
    var params = [];
    var exeName = sanitizeForWatson(exe);
    var thisVersion = advanced_1.convertVersion(thisPackage.version);
    var p1Value = exeName.concat("_", thisVersion, "_x86__");
    var param1 = { paramNumber: 0, paramValue: p1Value, paramName: "Package Name" };
    var param2 = { paramNumber: 1, paramValue: thisPackage.name, paramName: "Application Name" };
    var param3 = {
        paramNumber: 2,
        paramValue: "0x" + knownParams.stackHash.substr(0, 4),
        paramName: "Stack Hash" };
    var param4 = { paramNumber: 3, paramValue: "0x00000000", paramName: "Error Type" };
    var param5 = { paramNumber: 4, paramValue: knownParams.sourceFile, paramName: "Source File URL" };
    var lineHexString = knownParams.line.toString(16);
    var colHexString = knownParams.column.toString(16);
    lineHexString = eightZeroFill(lineHexString);
    colHexString = eightZeroFill(colHexString);
    var param6 = {
        paramNumber: 5,
        paramValue: "0x" + lineHexString + colHexString,
        paramName: "Source Position"
    };
    params.push(param1, param2, param3, param4, param5, param6);
    return params;
}
exports.setParamsWWAJSE = setParamsWWAJSE;
var EIGHT_ZEROES = "00000000";
function eightZeroFill(input) {
    var stringInput = input.toString();
    if (stringInput.length >= EIGHT_ZEROES.length) {
        return stringInput;
    }
    return (EIGHT_ZEROES + stringInput).slice(-EIGHT_ZEROES.length);
}
function sanitizeForWatson(input) {
    return advanced_1.replaceAll(input, "_", "-");
}

// SIG // Begin signature block
// SIG // MIIh7wYJKoZIhvcNAQcCoIIh4DCCIdwCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // P4pc/4T4qvKu8nm0YHGHIQPEL47pYQil1XPWUbOU35Gg
// SIG // gguEMIIFDDCCA/SgAwIBAgITMwAAAU/nxmLJRvSpfwAA
// SIG // AAABTzANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTE2MTExNzIxNTkxNFoX
// SIG // DTE4MDIxNzIxNTkxNFowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBALSJkIp2DKzFPaG8gejXUAi7
// SIG // WmmnrsfMuFu9OA8hUUYII9G9Iplztyjv9werK/lI96UU
// SIG // BalNJBttYZacg50zwCx5mfBd8+zVRN+lzVn6dq0GP8ZK
// SIG // A+bqLZQQwmZv/WLQUIrRIpEjl2U0ocSaAl6S4rh8KHjC
// SIG // XuNG98rvOFeIrFYGWMqYG8eo1142rppFMOO2tz0Ywqcn
// SIG // sHlNU9gy5najUCr2P7q7QR4khk2Ym3qe9izivJBgZfuW
// SIG // afrgE2jiiSoudRidwm24IfSZXip/MaCnDczvTxFu/klC
// SIG // TmHeN8srnkDvzqtecAhzpXJ7t0bOWzJTXYo5UW+w16h4
// SIG // 3aMob17uQh8CAwEAAaOCAXswggF3MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjc9BgEGCCsGAQUFBwMDMB0GA1UdDgQWBBSS
// SIG // DqERcbYvVQ3DD/TJXHQuWgxqvzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwODY1
// SIG // K2I0YjEyODc4LWUyOTMtNDNlOS1iMjFlLTdkMzA3MTlk
// SIG // NDUyZjAfBgNVHSMEGDAWgBTm/F97uyIAWORyTrX0IXQj
// SIG // MubvrDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNDb2RTaWdQQ0FfMjAxMC0wNy0wNi5jcmwwWgYIKwYB
// SIG // BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNp
// SIG // Z1BDQV8yMDEwLTA3LTA2LmNydDAMBgNVHRMBAf8EAjAA
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQARwLlqQzX+hbE4gA24
// SIG // dHUsjXFJuDxB7S294gY2kHYrieW2gemG4xiMlpMyQqen
// SIG // 9rc0JTnQxnpD6tt3cMzU6ULrqMVVAnV4ntgQvgcbypRj
// SIG // +zwPce7UVWfXKfZqvwe85Y+C6vih4nMzPxSSXqG6ZVc3
// SIG // RYWAXNYsxos8aoejBxHL7NlxOCA1JD1UvcxVifR3EJvl
// SIG // maoPrsY8jp/aAE/PfeGzwCD3UjH3IkjVYY1/JgIZfYPM
// SIG // YZMq3bjc4gizp9/hBHdgyPSsNcsx5Wx1UcU3exNqTpVr
// SIG // j0CeyE0+rr1ZxSDWwnitt2RbO9DAFXA0FOXFnqYcx/T/
// SIG // 9stIweKN827OqrSgMIIGcDCCBFigAwIBAgIKYQxSTAAA
// SIG // AAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
// SIG // VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
// SIG // B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
// SIG // b3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3Qg
// SIG // Q2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAw
// SIG // NzA2MjA0MDE3WhcNMjUwNzA2MjA1MDE3WjB+MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQSAyMDEwMIIBIjANBgkqhkiG
// SIG // 9w0BAQEFAAOCAQ8AMIIBCgKCAQEA6Q5kUHlntcTj/QkA
// SIG // TJ6UrPdWaOpE2M/FWE+ppXZ8bUW60zmStKQe+fllguQX
// SIG // 0o/9RJwI6GWTzixVhL99COMuK6hBKxi3oktuSUxrFQfe
// SIG // 0dLCiR5xlM21f0u0rwjYzIjWaxeUOpPOJj/s5v40mFfV
// SIG // HV1J9rIqLtWFu1k/+JC0K4N0yiuzO0bj8EZJwRdmVMkc
// SIG // vR3EVWJXcvhnuSUgNN5dpqWVXqsogM3Vsp7lA7Vj07IU
// SIG // yMHIiiYKWX8H7P8O7YASNUwSpr5SW/Wm2uCLC0h31oVH
// SIG // 1RC5xuiq7otqLQVcYMa0KlucIxxfReMaFB5vN8sZM4Bq
// SIG // iU2jamZjeJPVMM+VHwIDAQABo4IB4zCCAd8wEAYJKwYB
// SIG // BAGCNxUBBAMCAQAwHQYDVR0OBBYEFOb8X3u7IgBY5HJO
// SIG // tfQhdCMy5u+sMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIA
// SIG // QwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/
// SIG // MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjE
// SIG // MFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
// SIG // b0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcB
// SIG // AQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0
// SIG // XzIwMTAtMDYtMjMuY3J0MIGdBgNVHSAEgZUwgZIwgY8G
// SIG // CSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9k
// SIG // ZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUA
// SIG // ZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBt
// SIG // AGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAGnTv
// SIG // V08pe8QWhXi4UNMi/AmdrIKX+DT/KiyXlRLl5L/Pv5PI
// SIG // 4zSp24G43B4AvtI1b6/lf3mVd+UC1PHr2M1OHhthosJa
// SIG // IxrwjKhiUUVnCOM/PB6T+DCFF8g5QKbXDrMhKeWloWmM
// SIG // IpPMdJjnoUdD8lOswA8waX/+0iUgbW9h098H1dlyACxp
// SIG // hnY9UdumOUjJN2FtB91TGcun1mHCv+KDqw/ga5uV1n0o
// SIG // UbCJSlGkmmzItx9KGg5pqdfcwX7RSXCqtq27ckdjF/qm
// SIG // 1qKmhuyoEESbY7ayaYkGx0aGehg/6MUdIdV7+QIjLcVB
// SIG // y78dTMgW77Gcf/wiS0mKbhXjpn92W9FTeZGFndXS2z1z
// SIG // NfM8rlSyUkdqwKoTldKOEdqZZ14yjPs3hdHcdYWch8Za
// SIG // V4XCv90Nj4ybLeu07s8n07VeafqkFgQBpyRnc89NT7be
// SIG // BVaXevfpUk30dwVPhcbYC/GO7UIJ0Q124yNWeCImNr7K
// SIG // sYxuqh3khdpHM2KPpMmRM19xHkCvmGXJIuhCISWKHC1g
// SIG // 2TeJQYkqFg/XYTyUaGBS79ZHmaCAQO4VgXc+nOBTGBpQ
// SIG // HTiVmx5mMxMnORd4hzbOTsNfsvU9R1O24OXbC2E9KteS
// SIG // LM43Wj5AQjGkHxAIwlacvyRdUQKdannSF9PawZSOB3sl
// SIG // cUSrBmrm1MbfI5qWdcUxghXDMIIVvwIBATCBlTB+MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3Nv
// SIG // ZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEwAhMzAAABT+fG
// SIG // YslG9Kl/AAAAAAFPMA0GCWCGSAFlAwQCAQUAoIGuMBkG
// SIG // CSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
// SIG // gjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJ
// SIG // BDEiBCCv0ir8c+ofeYVLdSdeRbjpU6ar8tZvqHJcaxlG
// SIG // gzR4ATBCBgorBgEEAYI3AgEMMTQwMqAYgBYASQBuAHQA
// SIG // ZQByAG4AYQBsAC4AagBzoRaAFGh0dHA6Ly9taWNyb3Nv
// SIG // ZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBAF5VsiVnNyk1
// SIG // k4ku6Jn+3Ol6VTgRpYe/nfftYKrVuIf0lGhMkv9knYxk
// SIG // XY44/uLUxU/zgHaq63bTVS1t02ALkvtAI5x23cEKQy5S
// SIG // yxhOOPVp+0aiH1395/lxR00qa3yp2QWOuWYuQ/Whnjw8
// SIG // B7ETDhKCFhl48BwkEkL97iw/VhpUFmPpcLJdhFNyl7iJ
// SIG // fZ85S6H7uQ7hYGaYJCKg1Vf3ySWuJZsfiG2NvyfvA46V
// SIG // HKHt6Mqq/Fs1zuE6FqRGjqOMDWqwD5TFzBpDkDIUJKPw
// SIG // Zi3Gq9mxoKTlvWqFO/yqKc7XUickA0qEV8wm/e00q3Xl
// SIG // SyX37+hj8Vjo52o5veqjmlahghNNMIITSQYKKwYBBAGC
// SIG // NwMDATGCEzkwghM1BgkqhkiG9w0BBwKgghMmMIITIgIB
// SIG // AzEPMA0GCWCGSAFlAwQCAQUAMIIBPQYLKoZIhvcNAQkQ
// SIG // AQSgggEsBIIBKDCCASQCAQEGCisGAQQBhFkKAwEwMTAN
// SIG // BglghkgBZQMEAgEFAAQgunjQdnNwrj85lQSC/spgGaZg
// SIG // 2yxGJN9bzKVQ1sf7+3MCBlg/QcsqqxgTMjAxNjEyMjgw
// SIG // MDU4MDYuMzkxWjAHAgEBgAIB9KCBuaSBtjCBszELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUG
// SIG // A1UECxMebkNpcGhlciBEU0UgRVNOOkJCRUMtMzBDQS0y
// SIG // REJFMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
// SIG // cCBTZXJ2aWNloIIO0DCCBnEwggRZoAMCAQICCmEJgSoA
// SIG // AAAAAAIwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTEw
// SIG // MDcwMTIxMzY1NVoXDTI1MDcwMTIxNDY1NVowfDELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
// SIG // IFRpbWUtU3RhbXAgUENBIDIwMTAwggEiMA0GCSqGSIb3
// SIG // DQEBAQUAA4IBDwAwggEKAoIBAQCpHQ28dxGKOiDs/BOX
// SIG // 9fp/aZRrdFQQ1aUKAIKF++18aEssX8XD5WHCdrc+Zitb
// SIG // 8BVTJwQxH0EbGpUdzgkTjnxhMFmxMEQP8WCIhFRDDNdN
// SIG // uDgIs0Ldk6zWczBXJoKjRQ3Q6vVHgc2/JGAyWGBG8lhH
// SIG // hjKEHnRhZ5FfgVSxz5NMksHEpl3RYRNuKMYa+YaAu99h
// SIG // /EbBJx0kZxJyGiGKr0tkiVBisV39dx898Fd1rL2KQk1A
// SIG // UdEPnAY+Z3/1ZsADlkR+79BL/W7lmsqxqPJ6Kgox8NpO
// SIG // BpG2iAg16HgcsOmZzTznL0S6p/TcZL2kAcEgCZN4zfy8
// SIG // wMlEXV4WnAEFTyJNAgMBAAGjggHmMIIB4jAQBgkrBgEE
// SIG // AYI3FQEEAwIBADAdBgNVHQ4EFgQU1WM6XIoxkPNDe3xG
// SIG // G8UzaFqFbVUwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBD
// SIG // AEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8w
// SIG // HwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQw
// SIG // VgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
// SIG // Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEB
// SIG // BE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNy
// SIG // b3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRf
// SIG // MjAxMC0wNi0yMy5jcnQwgaAGA1UdIAEB/wSBlTCBkjCB
// SIG // jwYJKwYBBAGCNy4DMIGBMD0GCCsGAQUFBwIBFjFodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vUEtJL2RvY3MvQ1BT
// SIG // L2RlZmF1bHQuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwA
// SIG // ZQBnAGEAbABfAFAAbwBsAGkAYwB5AF8AUwB0AGEAdABl
// SIG // AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQAH
// SIG // 5ohRDeLG4Jg/gXEDPZ2joSFvs+umzPUxvs8F4qn++ldt
// SIG // GTCzwsVmyWrf9efweL3HqJ4l4/m87WtUVwgrUYJEEvu5
// SIG // U4zM9GASinbMQEBBm9xcF/9c+V4XNZgkVkt070IQyK+/
// SIG // f8Z/8jd9Wj8c8pl5SpFSAK84Dxf1L3mBZdmptWvkx872
// SIG // ynoAb0swRCQiPM/tA6WWj1kpvLb9BOFwnzJKJ/1Vry/+
// SIG // tuWOM7tiX5rbV0Dp8c6ZZpCM/2pif93FSguRJuI57BlK
// SIG // cWOdeyFtw5yjojz6f32WapB4pm3S4Zz5Hfw42JT0xqUK
// SIG // loakvZ4argRCg7i1gJsiOCC1JeVk7Pf0v35jWSUPei45
// SIG // V3aicaoGig+JFrphpxHLmtgOR5qAxdDNp9DvfYPw4Ttx
// SIG // Cd9ddJgiCGHasFAeb73x4QDf5zEHpJM692VHeOj4qEir
// SIG // 995yfmFrb3epgcunCaw5u+zGy9iCtHLNHfS4hQEegPsb
// SIG // iSpUObJb2sgNVZl6h3M7COaYLeqN4DMuEin1wC9UJyH3
// SIG // yKxO2ii4sanblrKnQqLJzxlBTeCG+SqaoxFmMNO7dDJL
// SIG // 32N79ZmKLxvHIa9Zta7cRDyXUHHXodLFVeNp3lfB0d4w
// SIG // wP3M5k37Db9dT+mdHhk4L7zPWAUu7w2gUDXa7wknHNWz
// SIG // fjUeCLraNtvTX4/edIhJEjCCBNowggPCoAMCAQICEzMA
// SIG // AAChpf257qf8np0AAAAAAKEwDQYJKoZIhvcNAQELBQAw
// SIG // fDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcNMTYw
// SIG // OTA3MTc1NjQ4WhcNMTgwOTA3MTc1NjQ4WjCBszELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUG
// SIG // A1UECxMebkNpcGhlciBEU0UgRVNOOkJCRUMtMzBDQS0y
// SIG // REJFMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
// SIG // cCBTZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A
// SIG // MIIBCgKCAQEAm9ABeeYir3p8G3Ue87dn1h3ep94ANNgS
// SIG // +QfqCHsfTU3KhZR6q3ZrKgdFjVEn07ZdRqUlxmIUeYtP
// SIG // zOYs9eyfTXodNCI2KrjD4uzFUO3T/UPBLb/F8PrPzISQ
// SIG // 66Kmsm1XoI+5YXDUSc6IL4MuO4FKk7VJSsRlyZaF5C/6
// SIG // rOLYVx0z9r4Q58JSGxPg+RQ2qLOb9NsV8PTSa30tuFXO
// SIG // EelW/5TpIQ67kVfMnBV5cM2OrNPjgZmYww4H39tzxc8p
// SIG // Y/U+7DcYenP2JHW1/Mk3lDBXB9WgQBVNCxaw5tU3XTzY
// SIG // 06u8h5eHelVzS2FDwfMJiJK+zrjlhEo8FjecQc4gl4HI
// SIG // CQIDAQABo4IBGzCCARcwHQYDVR0OBBYEFKtcuYK+cSrV
// SIG // j+DosinP+hvTt/pIMB8GA1UdIwQYMBaAFNVjOlyKMZDz
// SIG // Q3t8RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0
// SIG // dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3By
// SIG // b2R1Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNy
// SIG // bDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0
// SIG // dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMv
// SIG // TWljVGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwGA1Ud
// SIG // EwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJ
// SIG // KoZIhvcNAQELBQADggEBACNgTLgFVOlnyb45PBUywxPI
// SIG // swCQfxczm/12L11MErPosCSi/rL0H6iyji5OEAdc6Pc0
// SIG // iu40HejhRIb4HtvePRKUh8GaD0Pgm/oUYau26hLjqohq
// SIG // 12V35Qdb0FBT0cVa1CgvKkpReR95OSp3x2HlI38qBdom
// SIG // ntVAtuJf3DoTdOU6/ar7PwL8K/n4IFJbKMpdsiAo7h0e
// SIG // 9IqEvBdS6rMScZosHRtODXjR25MNJF4XiElUIfzYXCbQ
// SIG // 6RPhbMpOvwe4O/nhnC9GDGU6nEWwCadzTCxrttcWY+D8
// SIG // cjiZpgXNMpFBol76u9etDnuFy/MPdzt4MtNPlpEUSCPG
// SIG // ipeXWB39pUGhggN5MIICYQIBATCB46GBuaSBtjCBszEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEn
// SIG // MCUGA1UECxMebkNpcGhlciBEU0UgRVNOOkJCRUMtMzBD
// SIG // QS0yREJFMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBTZXJ2aWNloiUKAQEwCQYFKw4DAhoFAAMVAIKu
// SIG // ifW05j8WXCC8F+TBw0DNOetooIHCMIG/pIG8MIG5MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScw
// SIG // JQYDVQQLEx5uQ2lwaGVyIE5UUyBFU046NTdGNi1DMUUw
// SIG // LTU1NEMxKzApBgNVBAMTIk1pY3Jvc29mdCBUaW1lIFNv
// SIG // dXJjZSBNYXN0ZXIgQ2xvY2swDQYJKoZIhvcNAQEFBQAC
// SIG // BQDcDYU+MCIYDzIwMTYxMjI4MDAyNjM4WhgPMjAxNjEy
// SIG // MjkwMDI2MzhaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIF
// SIG // ANwNhT4CAQAwCgIBAAICARQCAf8wBwIBAAICGv4wCgIF
// SIG // ANwO1r4CAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYB
// SIG // BAGEWQoDAaAKMAgCAQACAxbjYKEKMAgCAQACAwehIDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAQEAzW8yj6RjhC7xdjBvoXSl
// SIG // 4w44R2D+EHKcn++wIePpSkZ/mDwL3kV0SPGHy0+l5NFo
// SIG // Ebpf66C7Rlu8ifmwv2yhOhRzD0Sp4EkgvFNU+bm+ZdeB
// SIG // gdAP8VF79nHksM08FU4UETINaeFS/ZaA9UFQl1DHEvV0
// SIG // KFFXGRhi/4hLkIl/t7OGQexx4wNDSrnxdw0ej6iPMyql
// SIG // 3zCqPmh6ACbw88cSGF1czGP2uP7mYOzVfQiOiYzHe3aD
// SIG // VGPTjSUU90rs1+qahUSHh+vyhZK+lCct8lyqxxuUInQD
// SIG // +ehiOTkU11KTfPCdQa7nD31Sjfe5gIifUBJb28DmwUHz
// SIG // +HM8sp5UesznczGCAvUwggLxAgEBMIGTMHwxCzAJBgNV
// SIG // BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
// SIG // VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
// SIG // Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBU
// SIG // aW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAAoaX9ue6n/J6d
// SIG // AAAAAAChMA0GCWCGSAFlAwQCAQUAoIIBMjAaBgkqhkiG
// SIG // 9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkE
// SIG // MSIEIN2HhUen7caEf5jAvjSlEBTCrI3rHxHaP3Oj3wZV
// SIG // pR48MIHiBgsqhkiG9w0BCRACDDGB0jCBzzCBzDCBsQQU
// SIG // gq6J9bTmPxZcILwX5MHDQM0562gwgZgwgYCkfjB8MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
// SIG // ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAKGl/bnu
// SIG // p/yenQAAAAAAoTAWBBRLV4WfXe6QNv9iQniek3K3Q/9q
// SIG // MTANBgkqhkiG9w0BAQsFAASCAQBLdB54+TW096egyvW2
// SIG // 01G2chlKN7aXHlxTaNqug+LPJ1M3kwOrDGGOSA/dw0+h
// SIG // 69dV71cOPm0Ydn2QMGEORMZ4d56STjxMeKimMWac+gNr
// SIG // nEZRCBEAq4TySYBVWP3HVHVwau1B1VG59U+mAeA/lgGs
// SIG // usdfogiAFjGxWfi0fPhHYzQEV5rhFZNw1TchoVUKQPYr
// SIG // 3FymWJzr3TlOcUjiE0Lau5lQVbdQi37kwpa9JSQTrGDx
// SIG // cXNWN85oFRwDXI05oeII611A7htcQdeAUYU+lUPov6Jt
// SIG // ZWRPqnlYADz4Zh+yxvoSdANUaep3otlN6cq008uYf0cv
// SIG // /AZxS/ZD11UDh5i3
// SIG // End signature block
